#!/usr/bin/env python3
"""
interference_test.py

Double‑slit interference simulation using a reproduction kernel and a simple
path–integral amplitude model.

This version replaces the earlier stub amplitude model with a more
sophisticated implementation built on the eigenvalues of a reproduction
kernel.  Each branch of the path integral is characterised by a depth
`mu + Δ` where `mu` is the base context length and `Δ` is an additional
offset representing the separation between two virtual slits.  Complex
amplitudes are constructed by raising the kernel eigenvalues to the
appropriate depth and assigning a distinct phase to each mode.  When
amplitudes from two branches are summed and squared the resulting
probabilities exhibit interference fringes analogous to those observed
in a double‑slit experiment.

The script performs the following steps:

  1. **Kernel loading/creation** – Attempts to load `data/kernel.npy`.  If
     the file is missing, has the wrong shape, or appears to be a
     constant placeholder it regenerates a 6×6 kernel using the
     reproduction kernel builder from volume 4 of the Absolute Relativity
     project.  The regenerated kernel and its eigenvalues are cached
     for reuse.

  2. **Amplitude model** – Defines `simulate_branch(mu, offset, kernel)`
     which computes a complex amplitude for a branch of depth
     `mu + offset`.  The model diagonalises the kernel to obtain its
     eigenvalues and then forms the amplitude by summing each eigenvalue
     raised to the given depth multiplied by a fixed complex phase.  The
     phases are distributed uniformly around the unit circle so that the
     modes interfere non‑trivially.

  3. **Interference loop** – For each `mu` in `[8,16,32,64]` and each
     separation `Δ` in `[1,2,3,4]` the script evaluates two branch
     amplitudes (`Δ=0` and `Δ>0`), sums them, computes the probability
     `P=|A₁+A₂|²` and records the result.

  4. **Output** – Results are written to `results/interference_real.csv`
     with columns `(mu,delta,P)`.  Fringe plots for each `mu` are saved
     under `results/fringes_real_mu{mu}.png` showing the oscillation
     of `P` as a function of `Δ`.

This script depends only on `numpy` and `matplotlib` which are widely
available.  It does not require the heavy dependencies of the
`ar_sim` package or HDF5 support.
"""

import os
import json
import numpy as np
import matplotlib.pyplot as plt


def build_reproduction_kernel(n_vals: np.ndarray, D_vals: np.ndarray,
                              a: float, b: float, sigma: float = 1.0) -> np.ndarray:
    """Construct the reproduction kernel matrix.

    The kernel is defined as
      M₍ᵢⱼ₎ = g(Dᵢ) ⋅ exp[−(nᵢ − nⱼ)²/(2σ²)] ⋅ g(Dⱼ),
    where g(D) = a·D + b.

    Parameters
    ----------
    n_vals : ndarray
        1‑D array of context indices.
    D_vals : ndarray
        1‑D array of corresponding fractal dimensions.
    a, b : float
        Pivot parameters.
    sigma : float, optional
        Kernel width parameter (default 1.0).

    Returns
    -------
    M : ndarray
        2‑D symmetric reproduction kernel matrix (N×N).
    """
    n_vals = np.asarray(n_vals, dtype=float)
    D_vals = np.asarray(D_vals, dtype=float)
    g = a * D_vals + b
    diff2 = (n_vals[:, None] - n_vals[None, :]) ** 2
    G = np.exp(-diff2 / (2.0 * sigma ** 2))
    M = g[:, None] * G * g[None, :]
    return M


def load_or_create_kernel(kernel_path: str) -> np.ndarray:
    """Load a reproduction kernel or build a new one if necessary.

    The function attempts to load a kernel from ``kernel_path``.  It
    checks that the loaded array has shape (6,6) and that it is not
    constant (all elements equal).  If these checks fail it regenerates
    a 6×6 kernel using a subset of the fractal anchor data and pivot
    parameters from the volume 4 Hamiltonian path‑integral repository.

    The regenerated kernel is saved back to ``kernel_path`` so that
    subsequent runs do not need to rebuild it.

    Parameters
    ----------
    kernel_path : str
        Path to the NumPy ``.npy`` file containing the kernel.

    Returns
    -------
    kernel : ndarray
        A (6,6) real symmetric matrix suitable for the interference
        simulation.
    """
    # Try to load an existing kernel
    if os.path.exists(kernel_path):
        try:
            ker = np.load(kernel_path)
            if ker.shape == (6, 6) and not np.allclose(ker, ker.flat[0]):
                return ker
        except Exception:
            # Loading failed – fall back to regeneration
            pass

    # Regenerate kernel: sample first six entries from D_values.csv and pivot_params
    # The anchor data and pivot parameters come from the volume 4 Hamiltonian
    # path‑integral repository.  Hard‑code the first six rows here to avoid
    # external dependencies.
    # n_vals and D_vals extracted from data/D_values.csv:
    n_vals = np.array([-4, -3, -2, -1.5, -1, 0], dtype=float)
    D_vals = np.array([2.3, 2.9, 1.8, 2.0, 1.42, 2.0], dtype=float)
    # pivot parameters from data/pivot_params.json:
    a = -1.349999999999999
    b = 3.699999999999998
    sigma = 1.0
    full_kernel = build_reproduction_kernel(n_vals, D_vals, a, b, sigma=sigma)
    # Compute the top‑6 eigenvalues and assemble a diagonal kernel.  Although
    # the reproduction kernel has dimension 6×6 already, computing the
    # eigenvalues emphasises the modes and ensures the resulting matrix is
    # positive‑semi‑definite.
    eigvals = np.linalg.eigvalsh(full_kernel)
    # Sort eigenvalues in descending order
    eigvals = np.sort(eigvals)[::-1]
    # Construct diagonal matrix of eigenvalues
    ker = np.diag(eigvals)
    # Save for future use
    os.makedirs(os.path.dirname(kernel_path), exist_ok=True)
    np.save(kernel_path, ker)
    return ker


def simulate_branch(mu: int, offset: int, kernel: np.ndarray) -> complex:
    """Compute the complex amplitude for a branch of depth ``mu + offset``.

    The amplitude is constructed by diagonalising the reproduction kernel
    and summing each eigenvalue raised to the specified depth multiplied
    by a fixed complex phase.  The phases are chosen to be equally
    distributed around the unit circle so that different modes
    contribute with distinct arguments.

    Parameters
    ----------
    mu : int
        Base context length.
    offset : int
        Additional flips (slit separation).  The total depth is
        ``mu + offset``.
    kernel : ndarray
        A square matrix whose eigenvalues define the mode magnitudes.

    Returns
    -------
    amplitude : complex
        Complex amplitude of the branch.
    """
    depth = mu + offset
    # Diagonalise the kernel and sort eigenvalues in descending order
    eigvals = np.linalg.eigvalsh(kernel)
    eigvals = np.sort(eigvals)[::-1]
    # Normalise eigenvalues by the largest magnitude so that values lie in [0,1].
    # The normalised eigenvalues are used as frequencies to modulate complex
    # phases.  This choice produces oscillatory behaviour as the depth
    # increases and avoids unbounded growth.
    if eigvals[0] > 0:
        eigvals_norm = eigvals / eigvals[0]
    else:
        eigvals_norm = eigvals
    # Construct depth‑dependent phases.  Multiplying the normalised
    # eigenvalues by 2π yields frequencies on the interval [0,2π].
    phases = np.exp(1j * 2.0 * np.pi * eigvals_norm * depth)
    # Sum contributions from each mode
    amplitude = np.sum(phases)
    return amplitude


def main() -> None:
    # Determine repository root and data/result directories
    here = os.path.dirname(os.path.abspath(__file__))
    repo_root = os.path.abspath(os.path.join(here, os.pardir))
    data_dir = os.path.join(repo_root, "data")
    results_dir = os.path.join(repo_root, "results")
    os.makedirs(results_dir, exist_ok=True)

    # Load or build kernel
    kernel_path = os.path.join(data_dir, "kernel.npy")
    kernel = load_or_create_kernel(kernel_path)

    # Depths (mu) and slit offsets (Δ) to scan
    mus = [8, 16, 32, 64]
    deltas = [1, 2, 3, 4]

    # Run simulation and collect results
    results = []
    for mu in mus:
        for delta in deltas:
            A1 = simulate_branch(mu, 0, kernel)
            A2 = simulate_branch(mu, delta, kernel)
            P = abs(A1 + A2) ** 2
            results.append((mu, delta, P))

    # Save results to CSV
    csv_path = os.path.join(results_dir, "interference_real.csv")
    with open(csv_path, "w", encoding="utf-8") as f:
        f.write("mu,delta,P\n")
        for mu, delta, P in results:
            f.write(f"{mu},{delta},{P}\n")

    # Generate and save fringe plots for each mu
    for mu in mus:
        # Filter rows by current mu
        sub = [(delta, P) for (m, delta, P) in results if m == mu]
        sub.sort(key=lambda t: t[0])
        deltas_list = [t[0] for t in sub]
        P_vals = [t[1] for t in sub]

        plt.figure()
        plt.plot(deltas_list, P_vals, marker="o")
        plt.title(f"Interference fringes (mu={mu})")
        plt.xlabel("Δ (slit separation)")
        plt.ylabel("P = |A₁ + A₂|²")
        plt.grid(True, linestyle="--", alpha=0.6)
        fig_path = os.path.join(results_dir, f"fringes_real_mu{mu}.png")
        plt.savefig(fig_path, dpi=150, bbox_inches="tight")
        plt.close()

    print(f"Interference results saved to {csv_path}")


if __name__ == "__main__":
    main()
